import Stripe from "stripe";
import user from "../models/userModel.js";

const stripe = Stripe(process.env.STRIPE_SECRET_KEY);

const prices = {
  small: "price_1PQvRqP5RscoJ4WMTMKAdQNY",
  medium: "price_1PQvSqP5RscoJ4WM3icHBJgg",
  large: "price_1PQvTSP5RscoJ4WMZryHm28u",
};

const tokens = {
  small: 1000,
  medium: 5000,
  large: 10000,
};

//subscription management functions with stripe
export const createCustomer = async (name, email) => {
  try {
    const customer = await stripe.customers.create({
      name: name,
      email: email,
      payment_method: "pm_card_visa",
      invoice_settings: {
        default_payment_method: "pm_card_visa",
      },
    });
    return customer;
  } catch (e) {
    console.log("Error creating customer:", e.message);
    return null;
  }
};

const createSession = async (plan) => {
  try {
    console.log("Creating session for plan:", prices[plan]);
    const session = await stripe.checkout.sessions.create({
      payment_method_types: ["card"],
      line_items: [
        {
          price: prices[plan],
          quantity: 1,
        },
      ],
      mode: "subscription",
      // customer: customerId,
      success_url: `${process.env.CLIENT_URL}/my-plan?session_id={CHECKOUT_SESSION_ID}`,
      cancel_url: `${process.env.CLIENT_URL}/pricing?plan=${plan}&cancel=true`,
    });

    return session;
  } catch (e) {
    console.log("Error creating session:", e.message);
    return null;
  }
};

export const createCheckoutSession = async (req, res) => {
  try {
    const { plan, email } = req.body;

    // check if customer exists
    const customer = await stripe.customers.list({ email: email });

    if (customer.data.length === 0) {
      await createCustomer(plan, email);
    }
    // create checkout session
    const session = await createSession(plan);

    if (!session) {
      return res.send({
        status: "error",
        data: null,
        message: "Error creating checkout session!",
      });
    }

    console.log("Session created successfully:", session.url);

    const dev = await user.findOne({ email: email });

    if (!dev) {
      return res.send({
        status: "error",
        data: null,
        message: "User not found!",
      });
    }

    dev.subscription = {
      sessionId: session.id,
      priceId: prices[plan],
      plan: plan,
      customerId: customer.data[0].id,
      verified: false,
    };

    await dev.save();
    res.send({
      status: "success",
      data: session,
      message: "Checkout session created successfully!",
    });
  } catch (e) {
    console.log("Error creating checkout session:", e.message);
    res.send({
      status: "error",
      data: null,
      message: e.message,
    });
  }
};

export const paymentConfirm = async (req, res) => {
  try {
    const { sessionId } = req.body;
    const session = await stripe.checkout.sessions.retrieve(sessionId);
    const customer = await stripe.customers.retrieve(session.customer);

    console.log("customer", session);
    if (session.payment_status !== "paid") {
      return res.send({
        status: "error",
        data: null,
        message: "Payment not completed!",
      });
    }

    const dev = await user.findOne({ email: customer.email });

    if (!dev) {
      return res.send({
        status: "error",
        data: null,
        message: "User not found!",
      });
    }

    if (dev.subscription.verified) {
      return res.send({
        status: "success",
        data: { redirect: `${process.env.CLIENT_URL}/my-plan` },
        message: "Payment already confirmed!",
      });
    }

    dev.subscription = {
      verified: true,
    }; // update subscription status

    dev.tokens = dev.tokens + tokens[dev.subscription.plan]; // update token amount

    await dev.save();

    res.send({
      status: "success",
      data: {session: session, redirect: `${process.env.CLIENT_URL}/my-plan` },
      message: "Payment confirmed successfully!",
    });
  } catch (e) {
    console.log("Error confirming payment:", e.message);
    res.send({
      status: "error",
      data: null,
      message: e.message,
    });
  }
};

// create stripe product functions. will not use in the project, it's just for postman requests.
export const createProduct = async (req, res) => {
  try {
    const { name, description, price, currency } = req.body;
    const product = await stripe.products.create({
      name: name,
      description: description,
    });

    const prices = await stripe.prices.create({
      product: product.id,
      unit_amount: price,
      currency: currency,
      recurring: { interval: "year" },
    });
    res.send({
      status: "success",
      data: prices,
      message: "Product created successfully!",
    });
  } catch (e) {
    console.log("Error creating product:", e.message);
    res.send({
      status: "error",
      data: null,
      message: e.message,
    });
  }
};

export const productList = async (req, res) => {
  try {
    const products = await stripe.products.list();
    res.send({
      status: "success",
      data: products.data,
      message: "Products fetched successfully!",
    });
  } catch (e) {
    console.log("Error fetching products:", e.message);
    res.send({
      status: "error",
      data: null,
      message: e.message,
    });
  }
};

export const priceList = async (req, res) => {
  try {
    const prices = await stripe.prices.list();
    res.send({
      status: "success",
      data: prices.data,
      message: "Prices fetched successfully!",
    });
  } catch (e) {
    console.log("Error fetching prices:", e.message);
    res.send({
      status: "error",
      data: null,
      message: e.message,
    });
  }
};

export const removeProduct = async (req, res) => {
  try {
    const { productId } = req.body;
    await stripe.products.del(productId);
    res.send({
      status: "success",
      message: "Product deleted successfully!",
    });
  } catch (e) {
    console.log("Error deleting product:", e.message);
    res.send({
      status: "error",
      message: e.message,
    });
  }
};
