import Message from '../models/messageModel.js';
import Room from '../models/roomModel.js';
import User from '../models/userModel.js';
import {createAssistant, createThread, deleteThread, summarize} from "../providers/openai.js";

export const createRoom = async (req, res) => {
    const {roomName, context, isGroup, bots, participants} = req.body;
    console.log(roomName, bots, participants)
    if (!roomName || !context) {
        res.status(400).json({message: 'Please provide all the fields'});
    } else {
        try {
            let findRoom = await Room.find({
                roomName,
                groupAdmin: req.rootUserId,
            });
            if (findRoom.length > 0) {
                res.status(201).json({message: 'Room already exists'});
            } else {
                const assistant = await createAssistant(roomName);
                const thread = await createThread();
                if (!assistant || !thread) {
                    res.status(500).json({message: 'GPT assistant error'});
                } else {
                    const gpt = {
                        assistantId: assistant.id,
                        threadId: thread.id,
                    }
                    const room = await Room.create({
                        roomName,
                        context,
                        users: [req.rootUserId],
                        isGroup,
                        bots,
                        participants,
                        gpt,
                        groupAdmin: req.rootUserId
                    });
                    const finalRoom = await User.populate(room, {path: 'users', select: 'name email profilePic'});
                    res.status(200).json(finalRoom);
                }
            }
        } catch (error) {
            console.log(error);
            res.status(500).json({message: 'Something went wrong'});
        }
    }
}

export const fetchAllRooms = async (req, res) => {
    try {
        const rooms = await Room.find({users: {$elemMatch: {$eq: req.rootUserId}}})
            .populate('users', '-password')
            .populate('groupAdmin', '-password')
            .sort({updatedAt: -1});
        const finalRooms = await User.populate(rooms, {path: 'latestMessage.sender', select: 'name email profilePic'});
        res.status(200).json(finalRooms);
    } catch (error) {
        res.status(500).json({message: 'Something went wrong'});
        console.log(error);
    }
}

export const fetchRoomByFilter = async (req, res) => {
    const {roomIds} = req.body;
    console.log(req.body);
    try {
        let rooms = [];
        if (roomIds && roomIds.length > 0) {
            rooms = await Room.find({
                _id: {$in: roomIds},
                users: {$elemMatch: {$eq: req.rootUserId}},
            })
                .populate('users', '-password')
                .populate('groupAdmin', '-password')
                .sort({updatedAt: -1});
            const finalRooms = await User.populate(rooms, {
                path: 'latestMessage.sender',
                select: 'name email profilePic'
            });
            res.status(200).json(finalRooms);
        } else {
            res.status(200).json({message: 'No rooms found'});
        }
    } catch (error) {
        res.status(500).json({message: 'Something went wrong'});
        console.log(error);
    }

}

export const removeRoom = async (req, res) => {
    const {roomId} = req.body;
    console.log(roomId);
    if (!roomId) {
        res.status(400).json({message: 'Please provide room Id', status: 'error'});
    } else {
        try {
            const room = await Room.findById(roomId);
            if (!room) {
                res.status(400).json({message: 'Room does not exists', status: 'error'});
            } else {
                if (room.groupAdmin === req.rootUserId) {
                    res.status(400).json({message: 'You are not authorized to delete this room', status: 'error'});
                } else {
                    const threadId = room.gpt.threadId;
                    await Room.findByIdAndDelete(roomId);
                    await Message.deleteMany({roomId});
                    await deleteThread(threadId);
                    res.status(200).json({message: 'Room deleted successfully', status: 'success'});
                }
            }
        } catch (error) {
            res.status(500).json({message: 'Something went wrong', status: 'error'});
            console.log(error);
        }
    }

}

export const summarizeChat = async (req, res) => {
    const {roomId} = req.body;
    if (!roomId) {
        res.status(400).json({message: 'Please provide room Id', status: 'error'});
    } else {
        try {
            const room = await Room.findById(roomId);
            if (!room) {
                res.status(400).json({message: 'Room does not exists', status: 'error'});
            } else {
                const messages = await Message.find({roomId});
                for (let i = 0; i < messages.length; i++) {
                    const message = messages[i];
                    if (!message.star && message.type !== "duplicated") {

                        const summarized = await summarize(message.message);
                        message.summarized = summarized;
                        await message.save();
                    } else {
                        console.log("here ===== > ")
                        message.summarized = message.message;
                        await message.save();
                    }
                }

                //calculate token of subscription
                const dev = await User.findById(req.rootUserId);
                if (dev) {
                    dev.tokens = dev.tokens - 1;
                    await dev.save();
                }
                res.status(200).json({message: "Messages are summarized!", status: 'success'});
            }
        } catch (error) {
            res.status(500).json({message: 'Something went wrong', status: 'error'});
            console.log(error);
        }
    }
}